<?php
namespace WooCommerce_Contact_for_Shipping_Quote;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly


/**
 * Enqueue scripts.
 *
 * Enqueue script as javascript and style sheets.
 *
 * @since  1.0.0
 */
function enqueue_scripts() {

	wp_register_style( 'woocommerce-contact-for-shipping-quote', plugins_url( 'assets/front-end/css/style.min.css', \WooCommerce_Contact_for_Shipping_Quote\WooCommerce_Contact_For_Shipping_Quote()->file ), array(), \WooCommerce_Contact_for_Shipping_Quote\WooCommerce_Contact_For_Shipping_Quote()->version );
	wp_register_script( 'woocommerce-contact-for-shipping-quote', plugins_url( 'assets/front-end/js/main.js', \WooCommerce_Contact_for_Shipping_Quote\WooCommerce_Contact_For_Shipping_Quote()->file ), array(), \WooCommerce_Contact_for_Shipping_Quote\WooCommerce_Contact_For_Shipping_Quote()->version, true );

	wp_localize_script( 'woocommerce-contact-for-shipping-quote', 'wccsq', array(
		'ajaxurl' => admin_url( 'admin-ajax.php' ),
		'nonce'   => wp_create_nonce( 'contact-for-shipping-quote' ),
		'is_cart' => is_cart(),
	) );

	if ( is_cart() || is_checkout() ) {
		wp_enqueue_style( 'woocommerce-contact-for-shipping-quote' );
		wp_enqueue_script( 'woocommerce-contact-for-shipping-quote' );
	}
}
add_action( 'wp_enqueue_scripts', 'WooCommerce_Contact_for_Shipping_Quote\enqueue_scripts' );


/**
 * Get minimalistic cart data.
 *
 * Get a minimalistic set of cart data to store with the shipping quote.
 * We don't want to store a ton of data, so only store the data needed to provide a shipping quote.
 *
 * @since 1.0.0
 *
 * @param null $cart
 * @return array|null
 */
function get_cart_data( $cart = null ) {
	if ( is_null( $cart ) ) {
		$cart = WC()->cart->get_cart_contents();
	}

	$values = $cart;
	array_walk( $values, function( &$item ) {
		$item['name'] = $item['data']->get_title();
		$item = array_intersect_key( $item, array_flip( array( 'product_id', 'variation_id', 'variation', 'quantity', 'name', 'line_total', 'line_tax' ) ) );
	} );

	return apply_filters( 'WCCSQ/get_cart_data', array_values( $values ), $cart );
}


/**
 * Get the cart hash.
 *
 * Get a hash generated by the cart for easy validation in the database for a existing shipping quote.
 * Hash is based on product ID, variation ID, variation data and quantity.
 *
 * @since 1.0.0
 *
 * @param array $cart Cart items.
 * @return string Hashed cart.
 */
function get_cart_hash( $cart = null ) {
	$hash = md5( json_encode( get_cart_data( $cart ) ) );

	return apply_filters( 'WCCSQ/get_cart_hash', $hash, $cart );
}


/**
 * Get address data.
 *
 * Get the customer address data.
 *
 * @since 1.0.0
 *
 * @param array|null $address Customer address - can be partial values.
 * @return array Customer address.
 */
function get_address_data( $address = null ) {
	if ( is_null( $address ) ) {
		$address = wc_clean( array(
			'country'   => WC()->customer->get_shipping_country(),
			'postcode'  => WC()->customer->get_shipping_postcode(),
			'city'      => WC()->customer->get_shipping_city(),
			'address_1' => WC()->customer->get_shipping_address_1(),
			'address_2' => WC()->customer->get_shipping_address_2(),
		) );
	}

	// Ensure all required fields are present
	$address = wp_parse_args(
		$address,
		array( 'country' => null, 'state' => null, 'postcode' => null, 'city' => null, 'address_1' => '', 'address_2' => '' )
	);

	if ( ! empty( WC()->countries->get_states( $address['country'] ) ) ) {
		$address['state'] = wc_clean( (WC()->customer->get_shipping_state() ?: WC()->customer->get_billing_state()) );
	}

	return apply_filters( 'WCCSQ/get_address_data', $address );
}

/**
 * Get address hash.
 *
 * Get a hash specific to a address.
 *
 * @since 1.0.0
 *
 * @param array|null $address Address to hash. Leave empty to use checkout values.
 * @return string Address hash.
 */
function get_address_hash( $address = null ) {
	$address = get_address_data( $address );

	$hash = md5( json_encode( $address ) );

	return apply_filters( 'WCCSQ/get_address_hash', $hash, $address );
}


/**
 * Cart/customer recovery.
 *
 * Recover the cart items / customer address from a quote.
 *
 * @since 1.1.0
 */
function check_cart_recovery_action() {
	if ( ! isset( $_GET['action'], $_GET['quote'], $_GET['key'] ) || $_GET['action'] !== 'complete-quote' ) {
		return;
	}

	$quote_key = wc_clean( $_GET['key'] );
	$quote_id = absint( $_GET['quote'] );

	if ( ! $quote = get_shipping_quote( $quote_id ) ) {
		return;
	}

	$key_check = md5( $quote->get_cart_hash() . $quote->get_address_hash() );
	if ( $key_check !== $quote_key ) {
		return;
	}

	if ( ( ! isset( $_GET['force'] ) || $_GET['force'] != 1 ) && WC()->cart->get_cart_contents_count() !== 0 && get_cart_hash() !== $quote->get_cart_hash() ) {
		wc_add_notice(
			__( 'Your current cart is different from the items in the shipping quote.', 'woocommerce-contact-for-shipping-quote' ) .
			' <a href="' . esc_url( add_query_arg( 'force', 1, $quote->get_cart_recover_url() ) ) . '">' . __( 'Recover items from shipping quote.', 'woocommerce-contact-for-shipping-quote' ) . '</a>' );
		return;
	}

	// Recover the cart
	$quote->recover_cart();
	$quote->recover_customer();

	// Redirect after recovery
	wp_redirect( esc_url_raw( wc_get_cart_url() ) );
	die;
}
add_action( 'wp', '\WooCommerce_Contact_for_Shipping_Quote\check_cart_recovery_action' );
